"use client"

import { Button } from "@/components/ui/button"
import { Table, TableBody, TableCell, TableHead, TableHeader, TableRow } from "@/components/ui/table"
import { cn } from "@/lib/utils"
import { TransformedAgenciesData, TransformedAgency } from "@/types/transformers"
import { ComponentWithClassName, ComponentWithClassNameAndChildren, TODO } from "@/types/utils"
import { ChevronLeftIcon, ChevronRightIcon, SymbolIcon } from "@radix-ui/react-icons"
import { Column, ColumnPinningState, flexRender, getCoreRowModel, getPaginationRowModel, getSortedRowModel, Row, useReactTable, type SortingState, type Table as TanstackTable } from "@tanstack/react-table"
import { CSSProperties, useState } from "react"
import { columns } from "./agencies.table-columns"

export function AgenciesTable({ data, className }: ComponentWithClassName<{ data: NonNullable<TransformedAgenciesData> }>) {
  const [sorting, setSorting] = useState<SortingState>([])
  const [pagination, setPagination] = useState({
    pageIndex: 0,
    pageSize: 10,
  });
  const [columnPinning, setColumnPinning] = useState<ColumnPinningState>({
    left: ["agencyId"],
    right: [],
  });
  const table = useReactTable({
    data,
    columns,
    getCoreRowModel: getCoreRowModel(),
    getPaginationRowModel: getPaginationRowModel(),
    getSortedRowModel: getSortedRowModel(),
    onSortingChange: setSorting,
    onPaginationChange: setPagination,
    onColumnPinningChange: setColumnPinning,
    autoResetExpanded: false,
    autoResetPageIndex: false,
    enableColumnPinning: true,
    state: {
      sorting,
      pagination,
      columnPinning,
    },
    meta: {
      getRowStyles: (row: Row<TransformedAgency>): CSSProperties => ({
        background: row.original.access === "ALL" ? "red" : "blue"
      })

    }
  })

  return (
    <div className={cn("grid gap-3", className)}>
      <div className="grid gap-8">
        <SizableContentWrapper>
          <INTERNAL__DataTable
            table={table}
          />
        </SizableContentWrapper>
      </div>

      <div className="flex items-center justify-end space-x-2 py-4 p-6 pt-0">
        <div className="flex-1 text-sm text-muted-foreground">
          Showing{" "}
          {pagination.pageIndex + 1} of{" "}
          {table.getPageCount()} page(s).
        </div>

        <Button
          variant="ghost"
          size="icon"
          onClick={() => table.resetSorting()}
          className={sorting.length > 0 ? "text-muted-foreground" : "hidden"}
        >
          <div className="flex flex-row -space-x-2">
            <SymbolIcon className="size-4" />
          </div>
        </Button>
        <Button
          variant="ghost"
          size="icon"
          onClick={() => table.firstPage()}
          disabled={!table.getCanPreviousPage()}
        >
          <div className="flex flex-row -space-x-2">
            <ChevronLeftIcon className="size-4" />
            <ChevronLeftIcon className="size-4" />
          </div>
        </Button>
        <Button
          variant="ghost"
          size="icon"
          onClick={() => table.previousPage()}
          disabled={!table.getCanPreviousPage()}
        >
          <div className="flex flex-row -space-x-2">
            <ChevronLeftIcon className="size-4" />
          </div>
        </Button>
        <Button
          variant="ghost"
          size="icon"
          onClick={() => table.nextPage()}
          disabled={!table.getCanNextPage()}
        >
          <div className="flex flex-row -space-x-2">
            <ChevronRightIcon className="size-4" />
          </div>
        </Button>
        <Button
          variant="ghost"
          size="icon"
          onClick={() => table.lastPage()}
          disabled={!table.getCanNextPage()}
        >
          <div className="flex flex-row -space-x-2">
            <ChevronRightIcon className="size-4" />
            <ChevronRightIcon className="size-4" />
          </div>
        </Button>
      </div>
    </div>
  )
}

interface INTERNAL__DataTableProps<TData extends TransformedAgency> {
  table: TanstackTable<TData>
}

function INTERNAL__DataTable<TData extends TransformedAgency>({
  table
}: INTERNAL__DataTableProps<TData>) {

  return (
    <div>
      <div className="rounded-md border">
        <Table className="min-w-[64rem]">
          <TableHeader>
            {table.getHeaderGroups().map((headerGroup) => (
              <TableRow key={headerGroup.id}>
                {headerGroup.headers.map((header) => {
                  return (
                    <TableHead
                      className="[&:nth-child(2)]:pl-4 bg-background"
                      style={{ ...getCommonPinningStyles(header.column) }}
                      key={header.id}
                    >
                      {header.isPlaceholder
                        ? null
                        : flexRender(
                          header.column.columnDef.header,
                          header.getContext()
                        )}
                    </TableHead>
                  )
                })}
              </TableRow>
            ))}
          </TableHeader>
          <TableBody>
            {table.getRowModel().rows?.length ? (
              table.getRowModel().rows.map((row) => {
                console.log('row', row)
                return (
                  <TableRow
                    key={row.id}
                    className={row.original.access === "ALL" ? "[&>td:first-child]:pl-6 bg-background" : "[&>td:first-child]:pl-6 bg-muted"}
                    data-state={row.getIsSelected() && "selected"}
                  >
                    {row.getVisibleCells().map((cell) => (
                      <TableCell
                        key={cell.id}
                        className="[&:nth-child(2)]:pl-4 last-of-type:max-w-none"
                        style={{ ...getCommonPinningStyles(cell.column) }}
                      >
                        {flexRender(cell.column.columnDef.cell, cell.getContext())}
                      </TableCell>
                    ))}
                  </TableRow>
                )
              })
            ) : (
              <TableRow>
                <TableCell colSpan={columns.length} className="h-24 text-center">
                  <div
                    className={"w-full h-full border border-dashed rounded place-content-center border-border"}
                  >
                    <p className="text-xs text-muted-foreground text-center">
                      No results.
                    </p>
                  </div>
                </TableCell>
              </TableRow>
            )}
          </TableBody>
        </Table>
      </div>
    </div>
  )
}

function SizableContentWrapper({ className, children }: ComponentWithClassNameAndChildren) {
  return (
    <div className={cn("overflow-x-auto", className)}>
      {children}
    </div>
  )
}

const getCommonPinningStyles = (column: Column<TODO>): CSSProperties => {
  const isPinned = column.getIsPinned()
  const isLastLeftPinnedColumn =
    isPinned === 'left' && column.getIsLastColumn('left')
  const isFirstRightPinnedColumn =
    isPinned === 'right' && column.getIsFirstColumn('right')

  return {
    boxShadow: isLastLeftPinnedColumn
      ? '-1px 0 1px -1px gray inset'
      : isFirstRightPinnedColumn
        ? '1px 0 1px -1px gray inset'
        : undefined,
    left: isPinned === 'left' ? `${column.getStart('left')}px` : undefined,
    right: isPinned === 'right' ? `${column.getAfter('right')}px` : undefined,
    opacity: isPinned ? 0.95 : 1,
    position: isPinned ? 'sticky' : 'relative',
    width: column.getSize(),
    zIndex: isPinned ? 1 : 0,
  }
}
